<?php
/**
 * Elementor Widget
 * @package Bizmaster
 * @since 1.0.0
 */

namespace Elementor;
class Bizmaster_Pricing_One extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'bizmaster-pricing-one-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Pricing Table', 'bizmaster-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-price-table';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['bizmaster_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
		$this->start_controls_section(
			'settings_section',
			[
				'label' => esc_html__('General Settings', 'bizmaster-core'),
				'tab' => Controls_Manager::TAB_CONTENT
			]
		);

		$this->add_control(
			'price', [
				'label' => esc_html__('Price', 'bizmaster-core'),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__('enter price.', 'bizmaster-core'),
				'default' => esc_html__('$59', 'bizmaster-core')
			]
		);

		$this->add_control(
			'price_time', [
				'label' => esc_html__('Price Period', 'bizmaster-core'),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__('enter price period.', 'bizmaster-core'),
				'default' => esc_html__('/Monthly', 'bizmaster-core')
			]
		);

		$this->add_control(
			'sub_title', [
				'label' => esc_html__('Title', 'bizmaster-core'),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__('enter title.', 'bizmaster-core'),
				'default' => esc_html__('Consult', 'bizmaster-core')
			]
		);

		$this->add_control(
			'icon_image', [
				'label' => esc_html__('Image', 'bizmaster-core'),
				'type' => Controls_Manager::MEDIA,
				'description' => esc_html__('Upload image', 'bizmaster-core'),
				'default' => [
					'url' => Utils::get_placeholder_image_src()
				]
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
            'list_icon', [
                'label' => esc_html__('Icon', 'bizmaster-core'),
                'type' => Controls_Manager::ICONS,
				'description' => esc_html__('upload icon', 'bizmaster-core'),
				'default' => [
                    'value' => 'fas fa-angle-double-right',
                    'library' => 'solid',
                ]
            ]
        );

		$repeater->add_control(
			'list_title', [
				'label' => esc_html__('Title', 'bizmaster-core'),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__('enter title.', 'bizmaster-core'),
				'default' => esc_html__('Expert Car Repair Specialists', 'bizmaster-core')
			]
		);

		$this->add_control('pricing_items', [
			'label' => esc_html__('Pricing Items', 'bizmaster-core'),
			'type' => Controls_Manager::REPEATER,
			'fields' => $repeater->get_controls()
		]);

		$this->add_control(
			'price_btn', [
				'label' => esc_html__('Price Button Text', 'bizmaster-core'),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__('enter price button text.', 'bizmaster-core'),
				'default' => esc_html__('Get Started', 'bizmaster-core')
			]
		);

		$this->add_control(
			'price_btn_url',
			[
				'label' => esc_html__( 'Price Button Url', 'bizmaster-core' ),
				'type' => Controls_Manager::URL,
				'options' => [ 'url', 'is_external', 'nofollow' ],
				'description' => esc_html__('enter price button url.', 'bizmaster-core'),
				'default' => [
					'url' => '#',
					'is_external' => false,
					'nofollow' => false
				],
				'label_block' => true
			]
		);

		$this->end_controls_section();

        /*  tab styling tabs start */
		$this->start_controls_section(
            'pricing_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'bizmaster-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_control('pricing_card_bg_color', [
            'label' => esc_html__('Pricing Card BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .pricing-card" => "background: {{VALUE}}"
            ]
        ]);

		$this->add_responsive_control(
			'pricing_card_padding',
			[
				'label'      => esc_html__('Pricing Card Padding', 'bizmaster-core'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .pricing-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name'     => 'pricing_card_border',
				'label'    => esc_html__('Pricing Card Border', 'bizmaster-core'),
				'selector' => '{{WRAPPER}} .pricing-card'
			]
		);

		$this->add_control(
			'pricing_card_border_radius',
			[
				'label'      => esc_html__('Pricing Card Border Radius', 'bizmaster-core'),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .pricing-card' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control('price_color', [
			'label' => esc_html__('Price Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .pricing-card .pricing-card_price" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
			'label' => esc_html__('Price Typography', 'bizmaster-core'),
			'name' => 'price_typography',
			'description' => esc_html__('Price Typography', 'bizmaster-core'),
			'selector' => "{{WRAPPER}} .pricing-card .pricing-card_price"
		]);

		$this->add_control('price_period_color', [
			'label' => esc_html__('Price Period Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .pricing-card .pricing-card_price .duration" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
			'label' => esc_html__('Price Period Typography', 'bizmaster-core'),
			'name' => 'price_period_typography',
			'description' => esc_html__('Price Period Typography', 'bizmaster-core'),
			'selector' => "{{WRAPPER}} .pricing-card .pricing-card_price .duration"
		]);

		$this->add_control('subtitle_color', [
			'label' => esc_html__('Title Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .pricing-card .pricing-card_title" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
			'label' => esc_html__('Title Typography', 'bizmaster-core'),
			'name' => 'subtitle_typography',
			'description' => esc_html__('Title Typography', 'bizmaster-core'),
			'selector' => "{{WRAPPER}} .pricing-card .pricing-card_title"
		]);

		$this->add_control('list_icon_color', [
			'label' => esc_html__('List Icon Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .pricing-card .pricing-card-details .checklist li svg" => "color: {{VALUE}}; fill: {{VALUE}}",
				"{{WRAPPER}} .pricing-card .pricing-card-details .checklist li i" => "color: {{VALUE}}"
			]
		]);

		$this->add_control('list_title_color', [
			'label' => esc_html__('List Title Color', 'bizmaster-core'),
			'type' => Controls_Manager::COLOR,
			'selectors' => [
				"{{WRAPPER}} .pricing-card .pricing-card-details .checklist li" => "color: {{VALUE}}"
			]
		]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
			'label' => esc_html__('List Title Typography', 'bizmaster-core'),
			'name' => 'list_title_typography',
			'description' => esc_html__('List Title Typography', 'bizmaster-core'),
			'selector' => "{{WRAPPER}} .pricing-card .pricing-card-details .checklist li"
		]);

		$this->add_control('btn_color', [
            'label' => esc_html__('Button Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .pricing-card .pricing-card-details .global-btn" => "color: {{VALUE}}"
            ]
        ]);

		$this->add_control('btn_hover_color', [
            'label' => esc_html__('Button Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
				"{{WRAPPER}} .pricing-card .pricing-card-details .global-btn:hover" => "color: {{VALUE}}"
			]
		]);

		$this->add_control('btn_bg_color', [
            'label' => esc_html__('Button BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .pricing-card .pricing-card-details .global-btn" => "background: {{VALUE}}"
            ]
        ]);

		$this->add_control('btn_hover_bg_color', [
            'label' => esc_html__('Button BG Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .pricing-card .pricing-card-details .global-btn:after" => "background: {{VALUE}}",
				"{{WRAPPER}} .pricing-card .pricing-card-details .global-btn:before" => "background: {{VALUE}}"
            ]
        ]);

		$this->end_controls_section();
	}

    /**
     * Render Elementor widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$pricing_items = $settings['pricing_items'];
		?>
		<div class="pricing-card">
			<div class="pricing-card-price-wrap">
				<h3 class="pricing-card_price">
					<span class="currency"><?php echo $settings['price']; ?></span>
					<span class="duration"><?php echo $settings['price_time']; ?></span>
				</h3>
			</div>
			<div class="pricing-card-icon-wrap">
				<h4 class="pricing-card_title"><?php echo $settings['sub_title']; ?></h4>
				<div class="pricing-card_icon">
					<img src="<?php echo $settings['icon_image']['url']; ?>" alt="<?php echo $settings['sub_title']; ?>">
				</div>
			</div>
			<div class="pricing-card-details">
				<div class="checklist">
					<ul>
						<?php foreach ($pricing_items as $item): ?>
							<li>
								<?php Icons_Manager::render_icon($item['list_icon'], ['aria-hidden' => 'true']); ?>
								<?php echo $item['list_title']; ?>
							</li>
						<?php endforeach; ?>
					</ul>
				</div>
				<?php if(!empty($settings['price_btn'])) : ?>
					<?php
						if ( ! empty( $settings['price_btn_url']['url'] ) ) {
							$this->add_link_attributes( 'price_btn_url', $settings['price_btn_url'] );
						}
					?>
					<a class="global-btn" <?php $this->print_render_attribute_string( 'price_btn_url' ); ?>>
						<?php echo $settings['price_btn']; ?> <i class="fas fa-arrow-right ms-2"></i>
					</a>
				<?php endif; ?>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type(new Bizmaster_Pricing_One());
