<?php
/**
 * Elementor Widget
 * @package bizmaster
 * @since 1.0.0
 */

namespace Elementor;

class Bizmaster_Team_Grid_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'bizmaster-team-grid-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Team Grid', 'bizmaster-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['bizmaster_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
		$this->start_controls_section(
            'slider_settings_section',
            [
                'label' => esc_html__('Slider Settings', 'bizmaster-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'bizmaster-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many post you want in grid, enter -1 for unlimited post.')
        ]);

        $this->add_control('category', [
            'label' => esc_html__('Category', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => bizmaster()->get_terms_names('team-category', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'bizmaster-core'),
        ]);

        $this->add_control('order', [
            'label' => esc_html__('Order', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'bizmaster-core'),
                'DESC' => esc_html__('Descending', 'bizmaster-core'),
            ),
            'default' => 'ASC',
			'description' => esc_html__('Select Order', 'bizmaster-core')
        ]);

        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'bizmaster-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'bizmaster-core'),
                'title' => esc_html__('Title', 'bizmaster-core'),
                'date' => esc_html__('Date', 'bizmaster-core'),
                'rand' => esc_html__('Random', 'bizmaster-core'),
                'comment_count' => esc_html__('Most Comments', 'bizmaster-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'bizmaster-core')
        ]);

        $this->end_controls_section();

		$this->start_controls_section(
            'team_member_styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'bizmaster-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

		$this->add_control('team_card_bg_color', [
            'label' => esc_html__('Team Card BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-card" => "background-color: {{VALUE}}"
            ]
        ]);

		$this->add_control('share_icon_bg_color', [
            'label' => esc_html__('Share Icon BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-card .team-social .plus-btn" => "background-color: {{VALUE}}"
            ]
        ]);

		$this->add_control('share_icon_color', [
            'label' => esc_html__('Share Icon Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
			'default' => '#19352D',
			'selectors' => [
                "{{WRAPPER}} .team-card .team-social .plus-btn svg" => "fill: {{VALUE}}"
            ]
        ]);

        $this->add_control('social_share_icons_bg_color', [
            'label' => esc_html__('Social Share Icons BG Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-card .global-social" => "background-color: {{VALUE}}"
            ]
        ]);

        $this->add_control('social_share_icon_color', [
            'label' => esc_html__('Social Share Icon Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-social .global-social a" => "color: {{VALUE}}",
            ]
        ]);

		$this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-card_content h4.team-card_title a" => "color: {{VALUE}}",
            ]
        ]);

        $this->add_control('title_hover_color', [
            'label' => esc_html__('Title Hover Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-card_content h4.team-card_title a:hover" => "color: {{VALUE}}",
            ]
        ]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title Typography', 'bizmaster-core'),
            'name' => 'title_typography',
            'description' => esc_html__('Title Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .team-card_content h4.team-card_title a"
        ]);

		$this->add_control('subtitle_color', [
            'label' => esc_html__('Subtitle Color', 'bizmaster-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .team-card_content .team-card_desig" => "color: {{VALUE}}",
            ]
        ]);

		$this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Subtitle Typography', 'bizmaster-core'),
            'name' => 'subtitle_typography',
            'description' => esc_html__('Subtitle Typography', 'bizmaster-core'),
            'selector' => "{{WRAPPER}} .team-card_content .team-card_desig"
        ]);

        $this->end_controls_section();
	}

    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
	protected function render() {

		$settings = $this->get_settings_for_display();

        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];

        //setup query
        $args = array(
            'post_type' => 'team',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'team-category',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
		?>

        <div class="team-list-area">
			<div class="row">
                <?php
                while ($post_data->have_posts()) : $post_data->the_post();
                    $post_id = get_the_ID();
                    $img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
                    $img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'full', false) : '';
                    $img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
                    $img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);

					$team_single_meta_data = get_post_meta($post_id, 'bizmaster_team_options', true);
					$social_icons = isset($team_single_meta_data['social-icons']) && !empty($team_single_meta_data['social-icons']) ? $team_single_meta_data['social-icons'] : '';
				?>
                    <div class="col-lg-4">
                        <div class="team-card">
                            <div class="team-card_wrapp">
                                <div class="team-card_img">
                                    <img src="<?php echo esc_url($img_url) ?>" alt="<?php echo $img_alt; ?>">
                                </div>
                                <div class="team-social">
                                    <div class="plus-btn">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 14 14" fill="none">
                                            <path d="M14 3C14 4.65625 12.6562 6 11 6C10.1875 6 9.4375 5.6875 8.90625 5.1875L5.96875 6.65625C5.96875 6.75 5.96875 6.875 5.96875 7C5.96875 7.125 5.96875 7.25 5.96875 7.375L8.90625 8.84375C9.4375 8.34375 10.1875 8 11 8C12.6562 8 14 9.34375 14 11C14 12.6562 12.6562 14 11 14C9.3125 14 8 12.6562 8 11C8 10.875 8 10.75 8 10.6562L5.0625 9.1875C4.53125 9.6875 3.78125 10 3 10C1.3125 10 0 8.65625 0 7C0 5.34375 1.3125 4 3 4C3.78125 4 4.53125 4.34375 5.0625 4.84375L8 3.375C8 3.25 8 3.125 8 3C8 1.34375 9.3125 0 11 0C12.6562 0 14 1.34375 14 3ZM2.96875 8C3.53125 8 3.96875 7.5625 3.96875 7C3.96875 6.46875 3.53125 6 2.96875 6C2.4375 6 1.96875 6.46875 1.96875 7C1.96875 7.5625 2.4375 8 2.96875 8ZM11 2C10.4375 2 10 2.46875 10 3C10 3.5625 10.4375 4 11 4C11.5312 4 12 3.5625 12 3C12 2.46875 11.5312 2 11 2ZM11 12C11.5312 12 12 11.5625 12 11C12 10.4688 11.5312 10 11 10C10.4375 10 10 10.4688 10 11C10 11.5625 10.4375 12 11 12Z" />
                                        </svg>
                                    </div>
                                    <div class="global-social">
                                        <?php
                                            if (!empty($social_icons)) {
                                                foreach ($social_icons as $item) {
                                                    printf('<a href="%1$s"><i class="%2$s"></i></a>', $item['url'], $item['icon']);
                                                }
                                            }
                                        ?> 
                                    </div>
                                </div>
                            </div>
                            <div class="team-card_content">
                                <h4 class="team-card_title"><a href="<?php the_permalink() ?>"><?php echo esc_html(get_the_title($post_id)) ?></a></h4>
								<span class="team-card_desig"><?php echo $team_single_meta_data['designation'] ?></span>
							</div>
                        </div>
                    </div>
                <?php endwhile; ?>
            </div>
        </div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type(new Bizmaster_Team_Grid_Widget());
